/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.1.2.7 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright © 2007, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "global.h"
#include "intelpci.h"

static int iegd_gn4_fetch_size(void);
static void iegd_gn4_cleanup(void);
static void iegd_gn4_tlbflush(struct agp_memory *mem);
static int AGP_CREATE_GATT(iegd_gn4_create_gatt_table);

struct aper_size_info_fixed iegd_i965_sizes[] =
{
	/* VBIOS always allocates enough space for 512MB aperture */
	{128, 131072, 7},
	{64, 131072, 7},
	{256, 131072, 7},
	{512, 131072, 7},
};

bridge_driver_t drv_gn4 = {
       .owner                  = THIS_MODULE,
       .size_type              = FIXED_APER_SIZE,
       .aperture_sizes         = 0,
       .num_aperture_sizes     = 0,
       .needs_scratch_page     = TRUE,
       .configure              = iegd_cmn_configure,
       .fetch_size             = iegd_gn4_fetch_size,
       .cleanup                = iegd_gn4_cleanup,
       .tlb_flush              = iegd_gn4_tlbflush,
       .mask_memory            = iegd_cmn_mask_memory,
       .masks                  = iegd_cmn_masks,
       .agp_enable             = iegd_cmn_agp_enable,
       .cache_flush            = global_cache_flush,
       .create_gatt_table      = iegd_gn4_create_gatt_table,
       .free_gatt_table        = iegd_cmn_free_gatt_table,
       .insert_memory          = iegd_cmn_insert_entries,
       .remove_memory          = iegd_cmn_remove_entries,
       .alloc_by_type          = iegd_cmn_alloc_by_type,
       .free_by_type           = iegd_cmn_free_by_type,
       .agp_alloc_page         = agp_generic_alloc_page,
       .agp_destroy_page       = agp_generic_destroy_page,
};

static int iegd_gn4_fetch_size(void)
{
	struct aper_size_info_fixed *values;
	u32 offset = 0;
	u8 temp;

#define IG965_GMCH_MSAC 0x62
#define IGM965_GMCH_MSAC 0x66

	AGN_DEBUG("Enter");

	values = A_SIZE_FIX(agp_bridge->driver->aperture_sizes);

	if(private_data.pdev->device == PCI_DEVICE_ID_GM965) {
		pci_read_config_byte(private_data.pdev,
			IGM965_GMCH_MSAC, &temp);
	} else {
		pci_read_config_byte(private_data.pdev,
			IG965_GMCH_MSAC, &temp);
	}

	switch (temp & 6) {
	case 0:
		offset = 0;	/* 128MB aperture */
		break;
	case 2:
		offset = 2;	/* 256MB aperture */
		break;
	case 6:
		offset = 3;	/* 512MB aperture */
		break;
	}

	/* Set the actual size here */
	agp_bridge->previous_size = agp_bridge->current_size =
		(void *)(values + offset);

	AGN_DEBUG("Exit");

	/* Always return 512KB GTT when calculating available stolen memory */
	return values[3].size;
}

static void iegd_gn4_cleanup(void)
{
	AGN_DEBUG("Enter");
	iounmap((void *)private_data.registers);
	AGN_DEBUG("Exit");
}

static void iegd_gn4_tlbflush(struct agp_memory *mem)
{
	AGN_DEBUG("Enter");
	/* Gen4 must flush the GTT or simple 2D rendering will lock the engine. */
	writel(0, private_data.registers+0x2170);
	writel(0, private_data.registers+0x2174);
	AGN_DEBUG("Exit");
	return;
}

static int AGP_CREATE_GATT(iegd_gn4_create_gatt_table)
{
	const u32 i965_gtt_table_order = 7;

	int i;
	u16 j = 0;
	int num_entries;
	u32 gtt_bus_addr;
	u32 mmio_bus_addr;
	u32 gtt_enabled    = FALSE;
	u32 gtt_table_size = (1 << i965_gtt_table_order) * PAGE_SIZE - 1;
	u32 gtt_pgctl_reg;
	char *gtt_table, *gtt_table_end, *current_entry;
	struct page *gtt_table_page;

	AGN_DEBUG("Enter");

	agp_bridge->gatt_table_real = NULL;

	/* Find and save the address of the MMIO register */
	pci_read_config_dword(private_data.pdev, I915_MMADDR, &mmio_bus_addr);
	mmio_bus_addr &= 0xFFF80000;

	private_data.registers =(volatile u8 *) ioremap(mmio_bus_addr,1024 * 4096);
	if (!private_data.registers) {
		AGN_ERROR("ioremap failed to map");
		return (-ENOMEM);
	}

	/* GTT is mapped 512KB after the registers */
	private_data.gtt = (u32 __iomem *)((u32)private_data.registers +
		512*1024);

	/* Extract the content of the control register */
	gtt_pgctl_reg = readl(private_data.registers+I810_PGETBL_CTL);
	gtt_bus_addr  = gtt_pgctl_reg & 0xFFFFF000;
	gtt_enabled   = gtt_pgctl_reg & I810_PGETBL_ENABLED;

	global_cache_flush();
	agp_bridge->driver->tlb_flush(0);

	/* we have to call this as early as possible after the MMIO base address is known */
	iegd_cmn_init_gtt_entries();

	if( !gtt_enabled ) {
		num_entries   = intel_i830_sizes[0].num_entries;
		gtt_table     = (char *) __get_free_pages(GFP_KERNEL,
			i965_gtt_table_order);
		gtt_table_end = gtt_table + gtt_table_size;

		/* Make sure allocation was successful */
		if( NULL == gtt_table ) {
			AGN_ERROR("Fail to allocate kernel pages");
			return (-ENOMEM);
		}

		for( current_entry = gtt_table; current_entry < gtt_table_end;
		     current_entry += PAGE_SIZE ) {
		        gtt_table_page = virt_to_page( current_entry );
			set_bit( PG_reserved, &gtt_table_page->flags );
		}

		agp_bridge->gatt_bus_addr = virt_to_phys( gtt_table );

		for( i = 0; i < num_entries; i++ ) {
		  *(gtt_table + j) = (unsigned long) agp_bridge->scratch_page;
		  j += 4;
		}
	}
	else {
		agp_bridge->gatt_bus_addr = gtt_bus_addr;
	}

	agp_bridge->gatt_table = NULL;

	AGN_DEBUG("Exit");

	return(0);
}
