/*
 * WPA Supplicant - driver interaction with VNT 802.11a/b/g Linux driver
 * Copyright (c) 2005 Lyndon Chen <lyndonchen@vntek.com.tw> 
 * Copyright (c) 2003-2005, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/socket.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>   /* The L2 protocols */
#include <netinet/in.h>
#include <net/if.h>

#include "wireless_copy.h"
#include "common.h"
#include "driver.h"
#include "wpa_supplicant.h"
#include "eloop.h"
#include "wpa.h"
/* vnt driver: ../include */
#include "iocmd.h" 
#include "iowpa.h"

//2008-0714-01<Add>by MikeLiu
//mike:add for device attach&detach function
static int device_close;                                 //mike:add for device close event;
static char device_name[IFNAMSIZ + 1];

struct wpa_driver_viawget_data {
	void *ctx;
	char ifname[IFNAMSIZ + 1];
	int ioctl_sock;
	int event_sock;	
};


#define WPA_EVENT_MTU 2048


static int hostapd_ioctl_viawget(struct wpa_driver_viawget_data *drv,
				 struct viawget_wpa_param *param,
				 int len, int show_err)
{
	struct iwreq iwr;

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, drv->ifname, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) param;
	iwr.u.data.length = len;

	if (ioctl(drv->ioctl_sock, IOCTL_CMD_WPA, &iwr) < 0) {
		int ret = errno;
		if (show_err) 
			perror("ioctl[VIAWGET_IOCT]");
		return ret;
	}

	return 0;
}



/* This is called at wpa_supplicant daemon init time */
static int wpa_driver_viawget_set_wpa(void *priv, int enabled)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	size_t blen;
    int res;
		
	wpa_printf(MSG_DEBUG, "%s: %d ", __FUNCTION__, enabled);
   		
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;

	memset(param, 0, blen);
	param->cmd = VIAWGET_SET_WPA;
	param->u.generic_elem.len = 0;
	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	free(param);

	return res;
}


static int wpa_driver_viawget_set_key(void *priv, wpa_alg alg_name,
				      const u8 *addr, int key_idx, int set_tx,
				      const u8 *seq, size_t seq_len,
				      const u8 *key, size_t key_len)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	u8 *buf;
	size_t blen;
	int ret = 0;
	u8 key_array[64];

		   
	wpa_printf(MSG_DEBUG, "%s: alg=%d key_idx=%d set_tx=%d seq_len=%lu "
		   "key_len=%lu", __FUNCTION__, alg_name, key_idx, set_tx,
		   (unsigned long) seq_len, (unsigned long) key_len);		   

	wpa_hexdump(MSG_MSGDUMP, "addr=", addr, ETH_ALEN);
	wpa_hexdump(MSG_MSGDUMP, "key=", key, key_len);	

	blen = sizeof(*param);
	buf = malloc(blen);
	if (buf == NULL)
		return -1;
	memset(buf, 0, blen);
	
	memset(key_array, 0, 64);	
	if (key && key_len > 0) {
		memcpy(key_array, key, key_len);
		if (key_len == 32) {
		    // notice ! the oder 
			memcpy(&key_array[16], &key[24], 8);
			memcpy(&key_array[24], &key[16], 8);
		}
	}	
	
	param = (struct viawget_wpa_param *) buf;
	param->cmd = VIAWGET_SET_KEY;
	memcpy(param->addr, addr, ETH_ALEN);
	param->u.wpa_key.alg_name = (int)alg_name;
	param->u.wpa_key.set_tx = set_tx;
	param->u.wpa_key.key_index = key_idx;
	param->u.wpa_key.key_len = key_len;
	param->u.wpa_key.key = (u8 *)key_array; 
	param->u.wpa_key.seq = (u8 *)seq;
	param->u.wpa_key.seq_len = seq_len;

	if (hostapd_ioctl_viawget(drv, param, blen, 1)) {
		wpa_printf(MSG_WARNING, "Failed to set encryption.");
		ret = -1;
	}
	
	free(buf);

	return ret;
}


static int wpa_driver_viawget_set_countermeasures(void *priv,
						 int enabled)
{
	/* FIX */
	printf("wpa_driver_viawget_set_countermeasures - not yet "
	       "implemented\n");
	return 0;
}


static int wpa_driver_viawget_set_drop_unencrypted(void *priv,
						  int enabled)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	int res;
	size_t blen;
	
	wpa_printf(MSG_DEBUG, "%s: %d ", __FUNCTION__, enabled);	

	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;

	memset(param, 0, blen);
	param->cmd = VIAWGET_SET_DROP_UNENCRYPT;
	param->u.generic_elem.len = 0;
	res = hostapd_ioctl_viawget(drv, param, blen, 1);

	free(param);

	return res;
}


static int wpa_driver_viawget_deauthenticate(void *priv, const u8 *addr,
					     int reason_code)
{
	/* FIX */
	printf("wpa_driver_viawget_deauthenticate - not yet implemented\n");
	return 0;
}


static int wpa_driver_viawget_disassociate(void *priv, const u8 *addr,
					   int reason_code)
{

	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	int res = 0;
	size_t blen;
	
	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);
	wpa_hexdump(MSG_MSGDUMP, "addr=", addr, ETH_ALEN);	
	
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;
	memset(param, 0, blen);
    memcpy(param->addr, addr, ETH_ALEN);
	param->cmd = VIAWGET_SET_DISASSOCIATE;
	param->u.generic_elem.len = 0;
	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	free(param);

	return res;    	

}


static int
wpa_driver_viawget_associate(void *priv,
			     struct wpa_driver_associate_params *params)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	size_t blen = 0;
	int res = 0;		

//2006-1212-01<Add>by MikeLiu
        char temp_bssid[]={0x00,0x00,0x00,0x00,0x00,0x00};

    wpa_printf(MSG_DEBUG, "%s:  pairwise_suite = %d, group_suite = %d, key_mgmt_suite = %d auth_alg = %d",
    __FUNCTION__, params->pairwise_suite, params->group_suite, params->key_mgmt_suite, params->auth_alg);
                
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;

	memset(param, 0, blen);
	param->cmd = VIAWGET_SET_ASSOCIATE;	

 //2006-1212-02<Add>by MikeLiu
     if( params->bssid ==NULL)
	  memcpy(param->u.wpa_associate.bssid, temp_bssid, sizeof(temp_bssid));
	 else	 	
	  memcpy(param->u.wpa_associate.bssid, params->bssid, 6);
	param->u.wpa_associate.ssid_len = params->ssid_len;
	memcpy(param->u.wpa_associate.ssid, params->ssid, params->ssid_len);
	param->u.wpa_associate.wpa_ie = (u8 *)params->wpa_ie;
	param->u.wpa_associate.wpa_ie_len = params->wpa_ie_len;	
    param->u.wpa_associate.pairwise_suite = (int)params->pairwise_suite;
    param->u.wpa_associate.group_suite = (int)params->group_suite;    
    param->u.wpa_associate.key_mgmt_suite = (int)params->key_mgmt_suite;    
    param->u.wpa_associate.auth_alg = params->auth_alg;    
    param->u.wpa_associate.mode = params->mode;      
    param->u.wpa_associate.roam_dbm = params->roam_dbm; //DavidWang 
    
	res = hostapd_ioctl_viawget(drv, param, blen, 1);

	free(param);

	return res;      

}


static int wpa_driver_viawget_get_bssid(void *priv, u8 *bssid)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	size_t blen = 0;		
	int res = 0;
	
	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;
	memset(param, 0, blen);
	param->cmd = VIAWGET_GET_BSSID;	
	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	
	if (res == 0)
	    memcpy(bssid, param->u.wpa_associate.bssid, 6);	
	    
	free(param);

	return res;      
    
}


static int wpa_driver_viawget_get_ssid(void *priv, u8 *ssid)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	size_t blen = 0;
	int res = 0;
	int ssid_len = 0;
	
	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);	
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;
	memset(param, 0, blen);
	param->cmd = VIAWGET_GET_SSID;	
	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	if (res == 0 && param->u.wpa_associate.ssid_len != 0) { 
	    memcpy(ssid, param->u.wpa_associate.ssid, param->u.wpa_associate.ssid_len);	
	    ssid_len = param->u.wpa_associate.ssid_len;
	}    
	    
	free(param);

	return ssid_len;         
}



static void wpa_driver_viawget_scan_timeout(void *eloop_ctx,
					     void *timeout_ctx)
{
	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);	    
	wpa_supplicant_event(timeout_ctx, EVENT_SCAN_RESULTS, NULL);
}


static int wpa_driver_viawget_scan(void *priv, const u8 *ssid, size_t ssid_len)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	int res = 0;
	size_t blen;
	

	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;

	memset(param, 0, blen);
	param->cmd = VIAWGET_SET_SCAN;

	//2007-0919-01<Add>by MikeLiu
/**set ap_scan=1&&scan_ssid=1 under hidden ssid mode**/
	//param->u.generic_elem.len = 0;
	memset(param->u.scan_req.ssid,0,sizeof(param->u.scan_req.ssid));
	memcpy(param->u.scan_req.ssid,ssid,(u8)ssid_len);
	param->u.scan_req.ssid_len = (u8)ssid_len;

	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	free(param);
	eloop_register_timeout(3, 0, wpa_driver_viawget_scan_timeout, drv,
			       drv->ctx);	
	return res;    
    
}

static int wpa_driver_viawget_get_scan_results(void *priv,
					    struct wpa_scan_result *results,
					    size_t max_size)
{
	struct wpa_driver_viawget_data *drv = priv;
	struct viawget_wpa_param *param;
	struct viawget_scan_result *scan_buf;
    int ap_num;		
	int res;    
	size_t blen;


	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);
	blen = sizeof(*param);
	param = (struct viawget_wpa_param *) malloc(blen);
	if (param == NULL)
		return -1;
	memset(param, 0, blen);

	//max 100 counts
	blen = sizeof(struct viawget_scan_result)* 100;
    param->u.scan_results.buf = malloc(blen);
   	if (param->u.scan_results.buf == NULL)
    	return -1;
    	
    memset(param->u.scan_results.buf, 0, blen);    
    
	blen = sizeof(*param);    	
    
	param->cmd = VIAWGET_GET_SCAN;

	res = hostapd_ioctl_viawget(drv, param, blen, 1);
	
	if (res < 0) 
        goto error_res;
	

	memset(results, 0, max_size * sizeof(struct wpa_scan_result));	
	
	scan_buf = (struct viawget_scan_result *)param->u.scan_results.buf;

	for (ap_num = 0; ap_num < param->u.scan_results.scan_count ; ++ap_num) {
		// todo
	    wpa_hexdump(MSG_MSGDUMP, "VIAWGET: buf",
			    (u8 *) scan_buf->bssid, ETH_ALEN);

		memcpy(results[ap_num].bssid, scan_buf->bssid, ETH_ALEN);
		memcpy(results[ap_num].ssid,  scan_buf->ssid, scan_buf->ssid_len);
		results[ap_num].ssid_len = scan_buf->ssid_len;
		results[ap_num].freq = scan_buf->freq;
		results[ap_num].caps = scan_buf->caps;;

		wpa_hexdump(MSG_MSGDUMP, "VIAWGET: AP WPA IEs",
			    (u8 *) scan_buf->wpa_ie, scan_buf->wpa_ie_len);
			    
		wpa_hexdump(MSG_MSGDUMP, "VIAWGET: AP RSN IEs",
			    (u8 *) scan_buf->rsn_ie, scan_buf->rsn_ie_len);			    

        memcpy(results[ap_num].wpa_ie, scan_buf->wpa_ie, scan_buf->wpa_ie_len);
        results[ap_num].wpa_ie_len = scan_buf->wpa_ie_len;
        memcpy(results[ap_num].rsn_ie, scan_buf->rsn_ie, scan_buf->rsn_ie_len);
        results[ap_num].rsn_ie_len = scan_buf->rsn_ie_len;

		scan_buf = (struct viawget_scan_result *) ((u8 *) scan_buf + sizeof(struct viawget_scan_result));
	}

	wpa_printf(MSG_MSGDUMP, "Received %d AP of scan results ",  ap_num);
		   
	res = ap_num;	

error_res:
		   
	free(param->u.scan_results.buf);
	free(param);

	return res;	
    
}

static int wpa_driver_set_iface_flags(struct wpa_driver_viawget_data *drv, const char *ifname, int dev_up)
{
	struct ifreq ifr;

	if (drv->ioctl_sock < 0)
		return -1;

	memset(&ifr, 0, sizeof(ifr));
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ);

	if (ioctl(drv->ioctl_sock, SIOCGIFFLAGS, &ifr) != 0) {
		perror("ioctl[SIOCGIFFLAGS]");
		return -1;
	}

	if (dev_up)
		ifr.ifr_flags |= IFF_UP;
	else
		ifr.ifr_flags &= ~IFF_UP;

	if (ioctl(drv->ioctl_sock, SIOCSIFFLAGS, &ifr) != 0) {
		perror("ioctl[SIOCSIFFLAGS]");
		return -1;
	}

	if (dev_up) {
		memset(&ifr, 0, sizeof(ifr));
		strncpy(ifr.ifr_name, ifname, IFNAMSIZ);
		ifr.ifr_mtu = WPA_EVENT_MTU;
		if (ioctl(drv->ioctl_sock, SIOCSIFMTU, &ifr) != 0) {
			perror("ioctl[SIOCSIFMTU]");
		}
	}

	return 0;
}

static int wpa_driver_setwpadev_down_flags(void)
{		
 int ioctl_sock;
 SCmdRequest	req;
 SCmdValue val;
   struct ifreq ifr;
        	ioctl_sock = socket(PF_INET, SOCK_DGRAM, 0);
	if (ioctl_sock < 0) 
		return  -1;
	//set wpadev flags down
        snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%swpa", "vnt");
        	if (ioctl(ioctl_sock, SIOCGIFFLAGS, &ifr) != 0) {
	    printf("wpa_driver_setwpadev_down_flags:fail at 1\n");
	         close(ioctl_sock);	
		return -1;
        }
			
         ifr.ifr_flags &= ~IFF_UP;
	if (ioctl(ioctl_sock, SIOCSIFFLAGS, &ifr) != 0) {
		printf("wpa_driver_setwpadev_down_flags:fail at 2\n");
		close(ioctl_sock);
		return -1;
	}
	//unregister wpadev
	memset(&req, 0, sizeof(SCmdRequest));
	strncpy((char *)req.name, device_name, IFNAMSIZ);
	val.dwValue = 0; 
	req.data = &val;	
         req.wCmdCode = WLAN_CMD_SET_WPA;
	if (ioctl(ioctl_sock, IOCTL_CMD_SET, &req) < 0) {
		printf("wpa_driver_setwpadev_down_flags:fail at 3\n");
		close(ioctl_sock);
		return -1;
	}
	
        close(ioctl_sock);
        device_close = 1;              //device has closed ahead;
     return 0;
}

static void wpa_driver_viawget_event_receive(int sock, void *ctx,
					      void *sock_ctx)
{
	char buf[8192];
	int left;
	viawget_wpa_header *wpahdr;
	union wpa_event_data data;
	
	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);	

	if ((left = recv(sock, buf, sizeof buf, 0)) == -1)
		return;

	wpa_hexdump(MSG_DEBUG, "RECEIVE EVENT",(u8 *)buf, left);

	if (left < sizeof(viawget_wpa_header))
		return;

		
    wpahdr = (viawget_wpa_header *) buf;

	memset(&data, 0, sizeof(data));

	switch (wpahdr->type) {
	case VIAWGET_ASSOC_MSG:
		left -= sizeof(viawget_wpa_header);
		wpa_printf(MSG_DEBUG, "VIAWGET: ASSOC MESSAGE (left: %d)",
			   left);
		if (left > 0) {
			data.assoc_info.req_ies = malloc(wpahdr->req_ie_len);
			if (data.assoc_info.req_ies == NULL)
				return;
			data.assoc_info.resp_ies = malloc(wpahdr->resp_ie_len);
			if (data.assoc_info.resp_ies == NULL)
				return;				
			data.assoc_info.req_ies_len = wpahdr->req_ie_len;			       
			data.assoc_info.resp_ies_len = wpahdr->resp_ie_len;			
			memcpy(data.assoc_info.req_ies,
			       buf + sizeof(viawget_wpa_header), wpahdr->req_ie_len);
			memcpy(data.assoc_info.resp_ies,
			       buf + sizeof(viawget_wpa_header) + wpahdr->req_ie_len, wpahdr->resp_ie_len);
			wpa_hexdump(MSG_MSGDUMP, "VIAWGET: copying %d bytes "
				    "into req_ies",
				    data.assoc_info.req_ies, left);
		}

		wpa_supplicant_event(ctx, EVENT_ASSOCINFO, &data);
		wpa_supplicant_event(ctx, EVENT_ASSOC, NULL);
		break;
	case VIAWGET_DISASSOC_MSG:	
		wpa_printf(MSG_DEBUG, "VIAWGET: DISASSOC MESSAGE");
		wpa_supplicant_event(ctx, EVENT_DISASSOC, NULL);
		break;
	case VIAWGET_PTK_MIC_MSG:			
		wpa_printf(MSG_DEBUG, "VIAWGET: PTK MIC MSG MESSAGE");
		data.michael_mic_failure.unicast = 1;
		wpa_supplicant_event(ctx, EVENT_MICHAEL_MIC_FAILURE, &data);
		break;
	case VIAWGET_GTK_MIC_MSG:
		wpa_printf(MSG_DEBUG, "VIAWGET: GTK MIC MSG MESSAGE");
		data.michael_mic_failure.unicast = 0;
		wpa_supplicant_event(ctx, EVENT_MICHAEL_MIC_FAILURE, &data);
		break;
           //DavidWang		
	case VIAWGET_CCKM_ROAM_MSG:
		wpa_printf(MSG_DEBUG, "VIAWGET: Cisco CCKM Roaming MESSAGE");
		wpa_supplicant_event(ctx, EVENT_CCKM_ROAM, &data);
              break;
           //DavidWang	

	//mike:add for device close event receive
        case VIAWGET_DEVICECLOSE_MSG:
                  wpa_printf(MSG_DEBUG, "VIAWGET:get device close MESSAGE\n");
	      	if(wpa_driver_setwpadev_down_flags()!=0)
		 wpa_printf(MSG_DEBUG, "VIAWGET: set device down fail\n");
	         else
		 wpa_printf(MSG_DEBUG, "VIAWGET: set device down secessful\n");
		break;
	default:
		wpa_printf(MSG_DEBUG, "VIAWGET: UNKNOWN MESSAGE (%d)",
			   wpahdr->type);
		break;
	}
}	

static void * wpa_driver_viawget_init(void *ctx, const char *ifname)
{
	struct wpa_driver_viawget_data *drv;
	struct sockaddr_ll addr_ll;
	struct ifreq ifr;		
	SCmdRequest	req;
	SCmdValue val;	

	wpa_printf(MSG_DEBUG, "%s: ", __FUNCTION__);
	drv = malloc(sizeof(*drv));
	if (drv == NULL)
		return NULL;
		
	memset(drv, 0, sizeof(*drv));

	drv->ctx = ctx;
	strncpy(drv->ifname, ifname, sizeof(drv->ifname));
	drv->ioctl_sock = socket(PF_INET, SOCK_DGRAM, 0);
	if (drv->ioctl_sock < 0) {
		free(drv);
		return NULL;
	}	
    
    //register eth*wpa interface
	memset(&req, 0, sizeof(SCmdRequest));
	strncpy((char *)req.name, ifname, IFNAMSIZ);
	val.dwValue = 1; 
	req.data = &val;	
    req.wCmdCode = WLAN_CMD_SET_WPA;
	if (ioctl(drv->ioctl_sock, IOCTL_CMD_SET, &req) < 0) {
		perror("ioctl[IOCTL_CMD_SET]");
		close(drv->ioctl_sock);
		free(drv);		
		return NULL;
	}	    

    memset(&ifr, 0, sizeof(ifr));
    snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%swpa", "vnt");
    if (ioctl(drv->ioctl_sock, SIOCGIFINDEX, &ifr) != 0) {
	    perror("ioctl(SIOCGIFINDEX)");
	    return NULL;
    }

    drv->event_sock = socket(PF_PACKET, SOCK_RAW, ntohs(ETH_P_ALL));
	if (drv->event_sock < 0) {
		perror("socket(PF_PACKET, SOCK_RAW, ntohs(ETH_P_ALL))");
		close(drv->ioctl_sock);
		free(drv);
		return NULL;
	}
	
	if (wpa_driver_set_iface_flags(drv, ifr.ifr_name, 1)) {
	    perror("wpa_driver_set_iface_flags");	    
		close(drv->event_sock);	    
		close(drv->ioctl_sock);
		free(drv);	    
		return NULL;
	}	

	memset(&addr_ll, 0, sizeof(addr_ll));
	addr_ll.sll_family = AF_PACKET;
	addr_ll.sll_ifindex = ifr.ifr_ifindex;
	addr_ll.sll_protocol = ntohs(ETH_P_ALL);		
	addr_ll.sll_hatype = 0;
	addr_ll.sll_pkttype = PACKET_HOST;
	addr_ll.sll_halen = 0;	

	if (bind(drv->event_sock, (struct sockaddr *) &addr_ll, sizeof(addr_ll)) < 0) {
		perror("bind(netlink)");
		close(drv->event_sock);
		close(drv->ioctl_sock);
		free(drv);
		return NULL;
	}

//mike:init device close event
         strncpy(device_name, ifname, sizeof(device_name));
         device_close = 0;

	eloop_register_read_sock(drv->event_sock, wpa_driver_viawget_event_receive, ctx,
				 NULL);
	
	return drv;
}


static void wpa_driver_viawget_deinit(void *priv)
{
	struct wpa_driver_viawget_data *drv = priv;
	SCmdRequest	req;
	SCmdValue val;
    struct ifreq ifr;	
		
if(device_close ==0) {
    // unregister netdev ethxwpa
    snprintf(ifr.ifr_name, sizeof(ifr.ifr_name), "%swpa", "vnt");
    wpa_driver_set_iface_flags(drv, ifr.ifr_name, 0);
	memset(&req, 0, sizeof(SCmdRequest));
	strncpy((char *)req.name, drv->ifname, IFNAMSIZ);
	val.dwValue = 0; 
	req.data = &val;	
    req.wCmdCode = WLAN_CMD_SET_WPA;
	if (ioctl(drv->ioctl_sock, IOCTL_CMD_SET, &req) < 0) {
		perror("ioctl[IOCTL_CMD_SET]");
	}	    		
   }
else
   printf("wpa_driver_viawget_deinit:device has close\n");
    wpa_driver_set_iface_flags(drv, drv->ifname, 0);
	eloop_unregister_read_sock(drv->event_sock);
	close(drv->event_sock);
	close(drv->ioctl_sock);
	free(drv);
}


struct wpa_driver_ops wpa_driver_viawget_ops = {
	.name = "vntwusb",
	.desc = "VNT 802.11a/b/g driver ",
	.get_bssid = wpa_driver_viawget_get_bssid,
	.get_ssid = wpa_driver_viawget_get_ssid,
	.set_wpa = wpa_driver_viawget_set_wpa,
	.set_key = wpa_driver_viawget_set_key,
	.set_countermeasures = wpa_driver_viawget_set_countermeasures,
	.set_drop_unencrypted = wpa_driver_viawget_set_drop_unencrypted,
	.scan = wpa_driver_viawget_scan,
	.get_scan_results = wpa_driver_viawget_get_scan_results,
	.deauthenticate = wpa_driver_viawget_deauthenticate,
	.disassociate = wpa_driver_viawget_disassociate,
	.associate = wpa_driver_viawget_associate,
	.init = wpa_driver_viawget_init,
	.deinit = wpa_driver_viawget_deinit,
};
